/**********************************
 *    generate.c                  *
 *    Colin Frayn                 *
 *    June 2002                   *
 **********************************/

/*
  This file contains all the functions for generating and testing
  random positions
*/

#include "stdafx.h"
#include "common.h"
#include "generate.h"
#include "board.h"
#include "checks.h"
#include "moves.h"
#include "resource.h"

extern BITBOARD RookMask[64], KnightMoves[64], BishopMask[64], FileMask[8];
extern BOOL bCheck, bCapt, bOneSol, bPromote;
extern HWND hWndMain;
char Solutions[FILENAME_MAX],**SolList=NULL;

/* Generate random board positions and test to see if they're
 * mate in 'n' problems */
int Generate(int *pieces, int npos, int *nfound, int ntry, int depth) {
  int n,shortest,newpos=0, sollength;
  char FEN[128];
  Board B;

   // Generate the positions
  for (n=*nfound;n<npos;n++) {
    // Try to find a position
    do {
      ntry++;
      newpos++;

      // Generate a new position
      GenerateRandomPosition(&B,pieces);

      // Test to see if it's a mate in 'n'
      shortest = TestMates(&B,(depth*2)-1,0);

      // Update position count
      if (ntry%10 == 0) SetDlgItemInt(hWndMain, IDC_NTESTED, ntry, FALSE);
        
       // Check for new input occasionally
      if (newpos > 20 && shortest != depth) return ntry;

      // If this is unsuccessful then try again!
    } while (shortest != (depth*2)-1);

    // Store this successful position in the list
    BoardToFEN(&B,FEN);
    (*nfound)++;
    SetDlgItemInt(hWndMain, IDC_NFOUND, *nfound, FALSE);
    SendDlgItemMessage(hWndMain, IDC_LIST, LB_ADDSTRING, 0, (LPARAM)FEN); 
    sollength = (int)strlen(Solutions);
    strncpy(SolList[(*nfound)-1], Solutions,sollength);
    SolList[(*nfound)-1][sollength] = 0;
  }
  return 0;
}


/* Convert a board position to a FEN notation line */
void BoardToFEN(Board *B, char *FEN) {
  int x,y,l=0,i=0,sq;
  char row[8];
  
  strcpy(FEN,"");
  // Loop through board squares
  for (y=0;y<8;y++) {
    i=l=0;
    strcpy(row,"");
    for (x=0;x<8;x++) {
      sq = (y*8)+x;
      if (B->pieces[sq]==0) l++;
      else {
        if (l>0) {row[i]=(char)(l+48);i++;}
        l=0;
        // Add the ID for this piece
        switch (B->pieces[sq]) {
         case (wpawn)  : row[i]='P'; break;
         case (wrook)  : row[i]='R'; break;
         case (wknight): row[i]='N'; break;
         case (wbishop): row[i]='B'; break;
         case (wqueen) : row[i]='Q'; break;
         case (wking)  : row[i]='K'; break;
         case (bpawn)  : row[i]='p'; break;
         case (brook)  : row[i]='r'; break;
         case (bknight): row[i]='n'; break;
         case (bbishop): row[i]='b'; break;
         case (bqueen) : row[i]='q'; break;
         case (bking)  : row[i]='k'; break;
        }
        i++;
      }
    }
    // Add the ID for a number of empty squares
    if (l>0) {row[i]=(char)(l+48);i++;}
    strncat(FEN,row,i);
    if (y<7) strcat(FEN,"/");
  }
   
  // Add the ID for "white to move, no castling capabilities, no EP square"
  strcat(FEN," w - -");
}

// Generate a random board position using the pieces specified
// Makes sure that the position is legal and the side to move isn't in check
void GenerateRandomPosition(Board *B, int *pieces) {
  int n,p,piecelist[64],sq;

  // Try to make a legal board
  do {
    // Reset the piece list
    memset(piecelist,0,sizeof(piecelist));
    // Loop through the pieces
    for (n=0;n<13;n++) {
      // Loop through the instances of this piece
      for (p=0;p<pieces[n];p++) {
        do {
          // Pick a legal square for this piece
          switch(n-6) {
            case wpawn : if (bPromote) sq = Random(40) + 16; else sq = Random(48)+8; break;
            case bpawn : if (bPromote) sq = Random(40) + 8;  else sq = Random(48)+8; break;
            default    : sq = Random(64); break;
          }
          // Check that this square is unoccupied
        } while (piecelist[sq]);
        // Put the piece here
        piecelist[sq] = n-6;
      }
    }
    // Setup this board
    SetBoard(B,piecelist);

    // Make sure nobody is in check
  } while (InCheck(B,WHITE) || InCheck(B,BLACK));
  // Hooray! We're done!
}

#define NO_MATE  (100)

// Return the shortest mate from this position (recursively)
int TestMates(Board *B, int depth, int ply) {
  int count=0,shortest=NO_MATE,disallowed=0,cm;
  char *solstr = Solutions;
  BOOL bBad = FALSE;
  // Move lists
  MOVE ml[MAX_MOVES], *m, *last;
  Undo U;

  // Generate moves
  last = GenerateMoves(B,B->side,ml);
  m = ml;

  // Loop through moves
  while (m<last) {

    // Do this move
    U = DoMove(B,*m);

    // Check for bad ply=0 moves
    if (ply==0) {
      bBad = FALSE;
      // First move gives check
      if (bCheck && GivesCheck(B,*m)) bBad = TRUE;
      // First move is a capture
      if (bCapt && U.capt) bBad = TRUE;
    }

     // Make sure that this isn't an illegal move into check
    cm = NO_MATE;
    if (!InCheck(B,Opponent(B->side))) {
      // Make sure this isn't a mate in 1
      if (GivesCheck(B,*m) && InCM(B,0)) cm=1;
      else if (depth>1) cm = TestMates(B,depth-1,ply+1) + 1;
      // If this is a shorter mate then update the scores
      if (cm < shortest) shortest = cm;
    }

    // Undo this move
    UndoMove(B,*m,U);

    // Abort if we've got a mate that's too short
    if (shortest < depth) return shortest;

    // If we have an escape move then return
    if (ply&1 && cm>NO_MATE) return cm;
      
    // Record the moves
    if (ply==0 && cm == depth) {
      if (bBad) return 0;
      // Abort if we have too many mates
      if (bOneSol && count) return 0;
      // Count this mate
      if (!bBad) {
        count++;
        if (count>1) *(solstr++) = ' ';
        solstr = MoveToText(solstr, *m, B);      
      }
    }

    // Goto next move
    m++;
  }
  // Finish off the solutions string
  if (ply==0) *solstr = (char)0;
  return shortest;
}

/* Pretty-print a move in PGN notation */
char *MoveToText(char *movestr, MOVE m, Board *B) {
  int to, from, chkpiece;
  Undo U;
  BITBOARD clash;

  if (m == NO_MOVE) {*movestr = 0;return movestr;}
  
  from=MFrom(m);
  to=MTo(m);
  clash = EMPTY;
  switch (PType(B->pieces[from])) {
   case pawn   : if (File(from)!=File(to)) *movestr++ = (char)(File(from) + 97); break;
   case knight : *movestr++ = 'N'; clash = B->WhiteKnights|B->BlackKnights; break;
   case bishop : *movestr++ = 'B'; clash = B->WhiteBishops|B->BlackBishops; break;
   case rook   : *movestr++ = 'R'; clash = B->WhiteRooks|B->BlackRooks; break;
   case queen  : *movestr++ = 'Q'; clash = B->WhiteQueens|B->BlackQueens; break;
   case king   : *movestr++ = 'K'; break;
  }

  /* Resolve Ambiguities */
  if (B->side==WHITE) clash &= B->WhitePieces;
  else clash &= B->BlackPieces;
  if (MoreThanOnePiece(clash)) {
    switch(PType(B->pieces[from])) {
     case rook  : clash &= (QueenMoves(B,to) & RookMask[to]); break;
     case knight: clash &= KnightMoves[to]; break;
     case bishop: clash &= (QueenMoves(B,to) & BishopMask[to]);
     case queen : clash &= QueenMoves(B,to); break;
    }
    if (MoreThanOnePiece(clash)) {
      if (OnePiece(clash&FileMask[File(from)]))
        *movestr++ = (char)(File(from)+97);
      else *movestr++ = (char)(56-Rank(from));
    }
  }
  if (IsEP(m) || B->pieces[to]!=empty) *movestr++ = 'x';
  *movestr++  = (char)(File(to)+97);
  *movestr++  = (char)(56-Rank(to));
  switch (IsPromote(m)) {
   case (1) : *movestr++  = 'q'; break;
   case (2) : *movestr++  = 'r'; break;
   case (3) : *movestr++  = 'n'; break;
   case (4) : *movestr++  = 'b'; break;
  }
  U = DoMove(B,m);
  if ((chkpiece = InCheck(B,B->side)) != FALSE) {
    if (InCM(B,chkpiece)) *movestr++ = '#';
    else *movestr++ = '+';
  }
  UndoMove(B,m,U);
  *movestr = 0;
  return movestr;
}
