/**********************************
 *    moves.c                     *
 *    Colin Frayn                 *
 *    June 2002                   *
 **********************************/

/*
  This file contains all the functions for generating list of
  possible moves from positions. 
*/

#include "stdafx.h"
#include "common.h"
#include "board.h"

extern int PromotePiece[5];
extern int DiagShifts_a1h8[64],DiagShifts_a8h1[64];
extern int DiagonalMask_a1h8[64],DiagonalMask_a8h1[64];
extern BITBOARD Mask[64],InvMask[64],FileMask[8],RankMask[8],QueenMask[64];
extern BITBOARD KnightMoves[64],KingMoves[64],MovesRank[64][256],MovesFile[64][256];
extern BITBOARD Movesa1h8[64][256],Movesa8h1[64][256];
extern BITBOARD IMR90[64],MR90[64],IMR45[64],IML45[64],MR45[64],ML45[64];

/* Generate all pseudolegal moves for specified side from
 * the specified board position.  Throw them in a list and then
 * forget about them. */
MOVE *GenerateMoves(const Board *B, const int side, MOVE *movelist) {
  int to,from=-1,i;
  MOVE tempmove;
  BITBOARD tsq=0,temp=0,mask;

  /* Generate moves for white pawns first */
  switch (side) {
  case WHITE :
    
    /* Promotions */
    tsq = ((B->WhitePawns >> 8) & FullRank) & ~(B->All);
    while (tsq) {
      to = FirstPiece(tsq);
      from = to + 8;
      tempmove = from + (to<<6);
      for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
      RemoveFirst(tsq);
    }
    /* Captures (including En-Passant) */
    tsq = (((B->WhitePawns & ~(FileMask[FileH])) >> 7) | ((B->WhitePawns & ~(FileMask[FileA])) >> 9));
    if (B->ep>=0) tsq &= (B->BlackPieces | Mask[B->ep]);
    else tsq &= B->BlackPieces;
    while (tsq) {
      to = FirstPiece(tsq);
      if (File(to)>0) {
        from = to + 7;
        if (B->WhitePawns & Mask[from]) {	
          tempmove = from + (to<<6);
          if (Rank(to)==0) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (!(B->BlackPieces & Mask[to])) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      if (File(to)<7) {
        from = to + 9;
        if (B->WhitePawns & Mask[from]) {
          tempmove = from + (to<<6);
          if (Rank(to)==0) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (!(B->BlackPieces & Mask[to])) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      RemoveFirst(tsq);
    }
    /* Pushes */
    /* One rank pushes */
    tsq = (B->WhitePawns >> 8) & ~(B->All);
    tsq &= ~RankMask[Rank8];
    /* Two rank initial pushes */
    temp = (((B->WhitePawns & RankMask[Rank2]) >> 16) & ~(B->All));
    /* Ensure that the 'step-over' square is empty too! */
    temp &= ~(B->All >> 8);
    /* Add them on */
    tsq |= temp;
    while (tsq) {
      to = FirstPiece(tsq);
      if (B->WhitePawns & Mask[to+8]) from = to + 8;
      else from = to + 16;
      *movelist++ = from + (to<<6);
      RemoveFirst(tsq);
    }
    
    break;
    
    /* ... or black pawns */
  case BLACK:
    
    /* Promotions */
    tsq = ((B->BlackPawns << 8) & RankMask[Rank1]) & ~(B->All);
    while (tsq) {
      to = FirstPiece(tsq);
      from = to - 8;
      tempmove = from + (to<<6);
      for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
      RemoveFirst(tsq);
    }
    /* Captures (including En-Passant) */
    tsq = (((B->BlackPawns & ~(FileMask[FileH])) << 9) | ((B->BlackPawns & ~(FileMask[FileA])) << 7));
    if (B->ep>=0) tsq &= (B->WhitePieces | Mask[B->ep]);
    else tsq &= B->WhitePieces;
    while (tsq) {
      to = FirstPiece(tsq);
      if (File(to)>0) {
        from = to - 9;
        if (B->BlackPawns & Mask[from]) {	
          tempmove = from + (to<<6);
          if (Rank(to)==7) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      if (File(to)<7) {
        from = to - 7;
        if (B->BlackPawns & Mask[from]) {
          tempmove = from + (to<<6);
          if (Rank(to)==7) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      RemoveFirst(tsq);
    }
    /* Pushes */
    /* One rank pushes */
    tsq = (B->BlackPawns << 8) & ~(B->All);
    tsq &= ~RankMask[Rank1];
    /* Two rank initial pushes */
    temp = (((B->BlackPawns & RankMask[Rank7]) << 16) & ~(B->All));
    /* Ensure that the 'step-over' square is empty too! */
    temp &= ~(B->All << 8);
    /* Add them on */
    tsq |= temp;
    while (tsq) {
      to = FirstPiece(tsq);
      if (B->BlackPawns & Mask[to-8]) from = to - 8;
      else from = to - 16;
      *movelist++ = from + (to<<6);
      RemoveFirst(tsq);
    }
    break;
  }

   /* Now Generate King Moves */  
  switch(side) {
   case WHITE:
    from = B->WhiteKing;
    tsq = KingMoves[from] & ~(B->WhitePieces);
    break;
   case BLACK:
    from = B->BlackKing;
    tsq = KingMoves[from] & ~(B->BlackPieces);
    break;
  }
   /* tsq holds a board of all possible target squares */
  while (tsq) {
    to = FirstPiece(tsq);
    *movelist++ = from + (to<<6); 
    RemoveFirst(tsq);
  }

   /* Now Generate Knight Moves */
  switch (side) {
   case WHITE: tsq = B->WhiteKnights; break;
   case BLACK: tsq = B->BlackKnights; break;
  }
   /* tsq holds a board of all possible knights */
  while (tsq) {
    from = FirstPiece(tsq);
    switch(side) {
     case WHITE: temp = KnightMoves[from] & ~(B->WhitePieces); break;
     case BLACK: temp = KnightMoves[from] & ~(B->BlackPieces); break;
    }
     /* temp holds a board of all possible target squares for this knight */
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6);
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Rook Moves */
  switch (side) {
   case WHITE: tsq = B->WhiteRooks; break;
   case BLACK: tsq = B->BlackRooks; break;
  }
   /* tsq now holds a board of all rooks */
  while (tsq) {
    from = FirstPiece(tsq);
     /* First generate horizontal moves */
    mask = (B->All >> (Rank(from)<<3)) & FullRank;
    temp = MovesRank[from][mask];
     /* Next generate vertical moves */
    mask = (B->R90 >> (File(from)<<3)) & FullRank;
    temp |= MovesFile[from][mask];
    switch(side) {
     case WHITE: temp &= ~(B->WhitePieces); break;
     case BLACK: temp &= ~(B->BlackPieces); break;
    }
     /* temp holds a board of all possible target squares */
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Bishop Moves
    * See above for comments, or look at my webpage
    * http://www.ast.cam.ac.uk/~cmf/chess/theory.html */
  switch (side) {
   case WHITE: tsq = B->WhiteBishops; break;
   case BLACK: tsq = B->BlackBishops; break;
  }
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    switch(side) {
     case WHITE: temp &= ~(B->WhitePieces); break;
     case BLACK: temp &= ~(B->BlackPieces); break;
    }
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Queen Moves */
  switch (side) {
   case WHITE: tsq = B->WhiteQueens; break;
   case BLACK: tsq = B->BlackQueens; break;
  }
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    mask = (B->All & RankMask[(Rank(from))]) >> (Rank(from)<<3);
    temp |= MovesRank[from][mask];
    mask = (B->R90 & RankMask[(File(from))]) >> (File(from)<<3);
    temp |= MovesFile[from][mask];
    switch(side) {
     case WHITE: temp &= ~(B->WhitePieces); break;
     case BLACK: temp &= ~(B->BlackPieces); break;
    }
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
   
   /* Return the last element so we know where we got up to in the move 
    * list i.e. how many moves there are to check */
  return movelist;
}

/* Generate all pseudolegal capture moves for specified side
 * from the specified board position.  This is useful for the
 * quiescence search */
MOVE *GenerateCaptures(const Board *B, const int side, MOVE *movelist) {
  int to,from,i;
  MOVE tempmove;
  BITBOARD tsq=0,temp,mask;
  
  /* Generate moves for white pawns first */
  if (side==WHITE) {
    
    /* Captures (including En-Passant) */
    tsq = (((B->WhitePawns & ~(FileMask[FileH])) >> 7) | ((B->WhitePawns & ~(FileMask[FileA])) >> 9));
    if (B->ep>0) tsq &= (B->BlackPieces | Mask[B->ep]);
    else tsq &= B->BlackPieces;
    while (tsq) {
      to = FirstPiece(tsq);
      if (File(to)>0) {
        from = to + 7;
        if (B->WhitePawns & Mask[from]) {	
          tempmove = from + (to<<6);
          if (Rank(to)==0) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      if (File(to)<7) {
        from = to + 9;
        if (B->WhitePawns & Mask[from]) {
          tempmove = from + (to<<6);
          if (Rank(to)==0) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      RemoveFirst(tsq);
    }
  }
  
  /* ... or black pawns */
  if (side==BLACK) {
    
    /* Captures (including En-Passant) */
    tsq = (((B->BlackPawns & ~(FileMask[FileH])) << 9) | ((B->BlackPawns & ~(FileMask[FileA])) << 7));
    if (B->ep>0) tsq &= (B->WhitePieces | Mask[B->ep]);
    else tsq &= B->WhitePieces;
    while (tsq) {
      to = FirstPiece(tsq);
      if (File(to)>0) {
        from = to - 9;
        if (B->BlackPawns & Mask[from]) {	
          tempmove = from + (to<<6);
          if (Rank(to)==7) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      if (File(to)<7) {
        from = to - 7;
        if (B->BlackPawns & Mask[from]) {
          tempmove = from + (to<<6);
          if (Rank(to)==7) for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
          else {
            /* Check for en-passant move */
            if (to==B->ep) tempmove |= EP_FLAG;
            /* Store move */
            *movelist++ = tempmove;
          }
        }
      }
      RemoveFirst(tsq);
    }
  }
  
  /* Now Generate King Moves */  
  if (side==WHITE) {
    from = B->WhiteKing;
    tsq = (KingMoves[from] & B->BlackPieces);
  }
  else {
    from = B->BlackKing;
    tsq = (KingMoves[from] & B->WhitePieces);
  }
  while (tsq) {
    to = FirstPiece(tsq);
    *movelist++ = from + (to<<6); 
    RemoveFirst(tsq);
  }
  
  /* Now Generate Knight Moves */
  if (side==WHITE) tsq = B->WhiteKnights;
  else             tsq = B->BlackKnights;
  while (tsq) {
    from = FirstPiece(tsq);
    if (side==WHITE) temp = KnightMoves[from] & B->BlackPieces;
    else             temp = KnightMoves[from] & B->WhitePieces;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6);
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
  
  /* Now Generate Rook Moves */
  if (side==WHITE) tsq = B->WhiteRooks;
  else tsq = B->BlackRooks;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = (B->All >> (Rank(from)<<3)) & FullRank;
    temp = MovesRank[from][mask];
    mask = (B->R90 >> (File(from)<<3)) & FullRank;
    temp |= MovesFile[from][mask];
    if (side==WHITE) temp &= B->BlackPieces;
    else             temp &= B->WhitePieces;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
  
  /* Now Generate Bishop Moves */
  if (side==WHITE) tsq = B->WhiteBishops;
  else tsq = B->BlackBishops;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    if (side==WHITE) temp &= B->BlackPieces;
    else             temp &= B->WhitePieces;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
  
  /* Now Generate Queen Moves */
  if (side==WHITE) tsq = B->WhiteQueens;
  else tsq = B->BlackQueens;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    mask = (B->All & RankMask[(Rank(from))]) >> (Rank(from)<<3);
    temp |= MovesRank[from][mask];
    mask = (B->R90 & RankMask[(File(from))]) >> (File(from)<<3);
    temp |= MovesFile[from][mask];
    if (side==WHITE) temp &= B->BlackPieces;
    else             temp &= B->WhitePieces;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
  
  /* Return the last element so we know where we got up to in the move 
    * list i.e. how many moves there are to check */
  return movelist;
}

/* Generate all pseudolegal moves for specified side blocking
 * sliding attacks to the specified square from the specified
 * board position. Ignore captures and king moves. This is used
 * for escaping checks. */
MOVE *GenerateBlockingMoves(const Board *B, const int side, const int target, MOVE *movelist) {
  int to,from,i,ty = Rank(target);
  MOVE tempmove;
  BITBOARD tsq=0,temp,mask,TargetMask = QueenMask[target],PieceTargetMask;

  PieceTargetMask = TargetMask & ~(B->All);
   
   /* Generate moves for white pawns first */
  if (side==WHITE) {
     
     /* Promotions */
    if (ty==0) tsq = ((B->WhitePawns >> 8) & FullRank) & ~(B->All);
    else tsq=0;
    while (tsq) {
      to = FirstPiece(tsq);
      from = to + 8;
      tempmove = from + (to<<6);
      for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
      RemoveFirst(tsq);
    }
     /* Pushes */
     /* One rank pushes */
    tsq = (B->WhitePawns >> 8) & ~(B->All);
    tsq &= ~RankMask[Rank8];
     /* Two rank initial pushes */
    temp = (((B->WhitePawns & RankMask[Rank2]) >> 16) & ~(B->All));
     /* Ensure that the 'step-over' square is empty too! */
    temp &= ~(B->All >> 8);
     /* Add them on */
    tsq |= temp;
     /* Mask them to generate only blockers */
    tsq &= TargetMask;
    while (tsq) {
      to = FirstPiece(tsq);
      if (B->WhitePawns & Mask[to+8]) from = to + 8;
      else from = to + 16;
      *movelist++ = from + (to<<6);
      RemoveFirst(tsq);
    }
     
  }

   /* ... or black pawns */
  if (side==BLACK) {
     
     /* Promotions */
    if (ty==7) tsq = ((B->BlackPawns << 8) & RankMask[Rank1]) & ~(B->All);
    else tsq=0;
    while (tsq) {
      to = FirstPiece(tsq);
      from = to - 8;
      tempmove = from + (to<<6);
      for (i=1;i<5;i++) *movelist++ = (tempmove + (i<<12));
      RemoveFirst(tsq);
    }
     /* Pushes */
     /* One rank pushes */
    tsq = (B->BlackPawns << 8) & ~(B->All);
    tsq &= ~RankMask[Rank1];
     /* Two rank initial pushes */
    temp = (((B->BlackPawns & RankMask[Rank7]) << 16) & ~(B->All));
     /* Ensure that the 'step-over' square is empty too! */
    temp &= ~(B->All << 8);
     /* Add them on */
    tsq |= temp;
     /* Mask them to generate only blockers */
    tsq &= TargetMask;
    while (tsq) {
      to = FirstPiece(tsq);
      if (B->BlackPawns & Mask[to-8]) from = to - 8;
      else from = to - 16;
      *movelist++ = from + (to<<6);
      RemoveFirst(tsq);
    }
  }

   /* Now Generate Knight Moves */
  if (side==WHITE) tsq = B->WhiteKnights;
  else             tsq = B->BlackKnights;
  while (tsq) {
    from = FirstPiece(tsq);
    if (side==WHITE) temp = KnightMoves[from] & PieceTargetMask;
    else             temp = KnightMoves[from] & PieceTargetMask;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6);
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Rook Moves */
  if (side==WHITE) tsq = B->WhiteRooks;
  else tsq = B->BlackRooks;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = (B->All >> (Rank(from)<<3)) & FullRank;
    temp = MovesRank[from][mask];
    mask = (B->R90 >> (File(from)<<3)) & FullRank;
    temp |= MovesFile[from][mask];
    if (side==WHITE) temp &= PieceTargetMask;
    else             temp &= PieceTargetMask;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Bishop Moves */
  if (side==WHITE) tsq = B->WhiteBishops;
  else tsq = B->BlackBishops;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    if (side==WHITE) temp &= PieceTargetMask;
    else             temp &= PieceTargetMask;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }

   /* Now Generate Queen Moves */
  if (side==WHITE) tsq = B->WhiteQueens;
  else tsq = B->BlackQueens;
  while (tsq) {
    from = FirstPiece(tsq);
    mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
    temp = Movesa1h8[from][mask];
    mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
    temp |= Movesa8h1[from][mask];
    mask = (B->All & RankMask[(Rank(from))]) >> (Rank(from)<<3);
    temp |= MovesRank[from][mask];
    mask = (B->R90 & RankMask[(File(from))]) >> (File(from)<<3);
    temp |= MovesFile[from][mask];
    if (side==WHITE) temp &= PieceTargetMask;
    else             temp &= PieceTargetMask;
    while (temp) {
      to = FirstPiece(temp);
      *movelist++ = from + (to<<6); 
      RemoveFirst(temp);
    }
    RemoveFirst(tsq);
  }
   
   /* Return the last element so we know where we got up to in the move 
    * list i.e. how many moves there are to check */
  return movelist;
}

       /* ----------======###    END OF MOVE GENERATION ROUTINES    ###======----------- */

/* Perform a move on the given Board.  Note - this routine does
 * NOT check to see if this move is legal!!! */
Undo DoMove(Board *B,const MOVE move) {
  int from=MFrom(move),to=MTo(move),promote=IsPromote(move), ep = IsEP(move);
  int pf=B->pieces[from],pt=B->pieces[to],side = B->side;
  BITBOARD M = Mask[to], MIM = (Mask[from] | M);
  Undo U;
  
  /* Set up the undo information */
  U.ep = B->ep;
  U.capt = 0;
  U.R90 = B->R90;
  U.R45 = B->R45;
  U.L45 = B->L45;
  
  /* Alter the necessary global position information */
  B->ep=-1; /* No en-passant square just yet... */
  B->side = Opponent(side);  /* Swap sides */

  /* Normal Move */
  if (promote==0) {
    /* Update the bitboards */
    if (side==WHITE) {
      B->WhitePieces ^= MIM;
      switch (pf) {
      case (wpawn)  : B->WhitePawns ^= MIM; if (to==from-16) B->ep = from-8; break;
      case (wrook)  : B->WhiteRooks ^= MIM; break;
      case (wknight): B->WhiteKnights ^= MIM; break;
      case (wbishop): B->WhiteBishops ^= MIM; break;
      case (wqueen) : B->WhiteQueens ^= MIM; break;
      case (wking)  : B->WhiteKing = to; break;
      }
      B->pieces[from] = empty;
      B->pieces[to] = pf;
      /* Check for Capture */
      switch (pt) {
      case (bpawn)  : B->BlackPieces ^= M;B->BlackPawns ^= M;U.capt=bpawn; break;
      case (brook)  : B->BlackPieces ^= M;B->BlackRooks ^= M;U.capt=brook; break;
      case (bknight): B->BlackPieces ^= M;B->BlackKnights ^= M;U.capt=bknight; break;
      case (bbishop): B->BlackPieces ^= M;B->BlackBishops ^= M;U.capt=bbishop; break;
      case (bqueen) : B->BlackPieces ^= M;B->BlackQueens ^= M;U.capt=bqueen; break;
      }
      /* Check for EP capture */
      if (ep) {
        B->BlackPieces &= InvMask[to+8];
        B->BlackPawns &= InvMask[to+8];
        B->pieces[to+8] = empty;
        U.capt = bpawn;
        B->R90 &= IMR90[to+8];
        B->R45 &= IMR45[to+8];
        B->L45 &= IML45[to+8];
      }
    }
    if (side==BLACK) {
      B->BlackPieces ^= MIM;
      switch (pf) {
      case (bpawn)  : B->BlackPawns ^= MIM; if (to==from+16) B->ep = from+8; break;
      case (brook)  : B->BlackRooks ^= MIM; break;
      case (bknight): B->BlackKnights ^= MIM; break;
      case (bbishop): B->BlackBishops ^= MIM; break;
      case (bqueen) : B->BlackQueens ^= MIM; break;
      case (bking)  : B->BlackKing = to; break;
      }
      B->pieces[from] = empty;
      B->pieces[to] = pf;
      /* Check for Capture */
      switch (pt) {
      case (wpawn)  : B->WhitePieces ^= M;B->WhitePawns ^= M;U.capt=wpawn; break;
      case (wrook)  : B->WhitePieces ^= M;B->WhiteRooks ^= M;U.capt=wrook; break;
      case (wknight): B->WhitePieces ^= M;B->WhiteKnights ^= M;U.capt=wknight; break;
      case (wbishop): B->WhitePieces ^= M;B->WhiteBishops ^= M;U.capt=wbishop; break;
      case (wqueen) : B->WhitePieces ^= M;B->WhiteQueens ^= M;U.capt=wqueen; break;
      }
      /* Check for EP capture */
      if (ep) {
        B->WhitePieces &= InvMask[to-8];
        B->WhitePawns &= InvMask[to-8];
        B->pieces[to-8] = empty;
        U.capt = wpawn;
        B->R90 &= IMR90[to-8];
        B->R45 &= IMR45[to-8];
        B->L45 &= IML45[to-8];
      }
    }
    
    /* Update Boards */
    B->All  = B->WhitePieces | B->BlackPieces;
    B->R90 &= IMR90[from];
    B->R90 |= MR90[to];
    B->R45 &= IMR45[from];
    B->R45 |= MR45[to];
    B->L45 &= IML45[from];
    B->L45 |= ML45[to];
    return U;
  }
  
  /* Promotion */
  if (promote) {
    /* Convert number in 0-4 form to the correct promotion piece */
    promote = PromotePiece[promote];
    B->pieces[from] = 0;
    /* Update the bitboards */
    if (side==WHITE) {
      B->WhitePieces ^= MIM;
      B->WhitePawns &= InvMask[from];
      B->pieces[to] = promote;
      /* Check for Capture */
      if (pt) {
        B->BlackPieces ^= M;
        switch (pt) {
        case (brook)   : B->BlackRooks ^= M; break;
        case (bknight) : B->BlackKnights ^= M; break;
        case (bbishop) : B->BlackBishops ^= M; break;
        case (bqueen)  : B->BlackQueens ^= M; break;
        }
        U.capt = pt;
      }
      /* Keep track of the promoted piece */
      switch (promote) {
      case (rook)   : B->WhiteRooks |= M; break;
      case (knight) : B->WhiteKnights |= M; break;
      case (bishop) : B->WhiteBishops |= M; break;
      case (queen)  : B->WhiteQueens |= M; break;
      }
    }
    if (side==BLACK) {
      B->BlackPieces ^= MIM;
      B->BlackPawns &= InvMask[from];
      B->pieces[to] = -promote;
      /* Check for Capture */
      if (pt) {
        B->WhitePieces ^= M;
        switch (pt) {
        case (wrook)   : B->WhiteRooks ^= M; break;
        case (wknight) : B->WhiteKnights ^= M; break;
        case (wbishop) : B->WhiteBishops ^= M; break;
        case (wqueen)  : B->WhiteQueens ^= M; break;
        }
        U.capt = pt;
      }
      /* Keep track of the promoted piece */
      switch (promote) {
      case (rook)   : B->BlackRooks |= M; break;
      case (knight) : B->BlackKnights |= M; break;
      case (bishop) : B->BlackBishops |= M; break;
      case (queen)  : B->BlackQueens |= M; break;
      }
    }
    /* Update Boards */
    B->All = B->WhitePieces | B->BlackPieces;
    B->R90 &= IMR90[from];
    B->R90 |= MR90[to];
    B->R45 &= IMR45[from];
    B->R45 |= MR45[to];
    B->L45 &= IML45[from];
    B->L45 |= ML45[to];
    return U;
  }
  
  return U;
}

/* Undo a move on the given Board.  Note - this routine 
 * assumes that the undo information is correct */
void UndoMove(Board *B, const MOVE move, const Undo U) {
  int from=MFrom(move),to=MTo(move),promote=IsPromote(move),ep = IsEP(move);
  int pf=B->pieces[to], pt=U.capt, side = Opponent(B->side);
  BITBOARD M = Mask[to], MIM = (Mask[from] | M);

  B->ep=U.ep; /* Restore old en-passant setting */
  B->side = side;  /* Swap sides back */
  B->R90 = U.R90;
  B->R45 = U.R45;
  B->L45 = U.L45;
   
   /* Normal Move */
  if (promote==0) {
    /* Update the bitboards */
    if (side==WHITE) {
      B->WhitePieces ^= MIM;
      switch (pf) {
      case (wpawn)  : B->WhitePawns ^= MIM; break;
      case (wrook)  : B->WhiteRooks ^= MIM; break;
      case (wknight): B->WhiteKnights ^= MIM; break;
      case (wbishop): B->WhiteBishops ^= MIM; break;
      case (wqueen) : B->WhiteQueens ^= MIM; break;
      case (wking)  : B->WhiteKing = from; break;
      }
      B->pieces[to] = pt;
      B->pieces[from] = pf;
      /* Check for Capture */
      if (pt && !ep) {
        B->BlackPieces |= M;
        switch (pt) {
        case (bpawn)  : B->BlackPawns |= M; break;
        case (brook)  : B->BlackRooks |= M; break;
        case (bknight): B->BlackKnights |= M; break;
        case (bbishop): B->BlackBishops |= M; break;
        case (bqueen) : B->BlackQueens |= M; break;
        }
      }
      /* Check for EP capture. */
      if (ep) {
        B->BlackPieces |= Mask[to+8];
        B->BlackPawns |= Mask[to+8];
        B->pieces[to+8] = bpawn;
        B->pieces[to] = empty;
      }
    }
    if (side==BLACK) {
      B->BlackPieces ^= MIM;
      switch (pf) {
      case (bpawn)  : B->BlackPawns ^= MIM; break;
      case (brook)  : B->BlackRooks ^= MIM; break;
      case (bknight): B->BlackKnights ^= MIM; break;
      case (bbishop): B->BlackBishops ^= MIM; break;
      case (bqueen) : B->BlackQueens ^= MIM; break;
      case (bking)  : B->BlackKing = from; break;
      }
      B->pieces[to] = pt;
      B->pieces[from] = pf;
      /* Check for Capture */
      if (pt && !ep) {
        B->WhitePieces |= M;
        switch (pt) {
        case (wpawn)  : B->WhitePawns |= M; break;
        case (wrook)  : B->WhiteRooks |= M; break;
        case (wknight): B->WhiteKnights |= M; break;
        case (wbishop): B->WhiteBishops |= M; break;
        case (wqueen) : B->WhiteQueens |= M; break;
        }
      }
      /* Check for EP capture. */
      if (ep) {
        B->WhitePieces |= Mask[to-8];
        B->WhitePawns |= Mask[to-8];
        B->pieces[to-8] = wpawn;
        B->pieces[to] = empty;
      }
    }
    B->All = B->WhitePieces | B->BlackPieces;
    return;
  }
  
  /* Promotion */
  if (promote) {
    promote = PType(pf);
    B->pieces[to] = pt;
    /* Update the bitboards */
    if (side==WHITE) {
      B->WhitePieces ^= MIM;
      B->WhitePawns |= Mask[from];
      B->pieces[from] = wpawn;
      /* Check for Capture */
      if (pt) {
        B->BlackPieces |= M;
        switch (pt) {
        case (brook)   : B->BlackRooks |= M; break;
        case (bknight) : B->BlackKnights |= M; break;
        case (bbishop) : B->BlackBishops |= M; break;
        case (bqueen)  : B->BlackQueens |= M; break;
        }
      }
      switch (promote) {
      case (rook)   : B->WhiteRooks &= ~M; break;
      case (knight) : B->WhiteKnights &= ~M; break;
      case (bishop) : B->WhiteBishops &= ~M; break;
      case (queen)  : B->WhiteQueens &= ~M; break;
      }
    }
    if (side==BLACK) {
      B->BlackPieces ^= MIM;
      B->BlackPawns |= Mask[from];
      B->pieces[from] = bpawn;
      /* Check for Capture */
      if (pt) {
        B->WhitePieces |= M;
        switch (pt) {
        case (wrook)   : B->WhiteRooks |= M; break;
        case (wknight) : B->WhiteKnights |= M; break;
        case (wbishop) : B->WhiteBishops |= M; break;
        case (wqueen)  : B->WhiteQueens |= M; break;
        }
      }
      switch (promote) {
      case (rook)   : B->BlackRooks &= ~M; break;
      case (knight) : B->BlackKnights &= ~M; break;
      case (bishop) : B->BlackBishops &= ~M; break;
      case (queen)  : B->BlackQueens &= ~M; break;
      }
    }
    B->All = B->WhitePieces | B->BlackPieces;
  }
  
  return;
}

/* Generate all queen moves for the specified square */
BITBOARD QueenMoves(const Board *B,int from) {
  BITBOARD Moves,mask;

  mask = ((B->R45 >> DiagShifts_a1h8[from]) & DiagonalMask_a1h8[from]);
  Moves = Movesa1h8[from][mask];
  mask = ((B->L45 >> DiagShifts_a8h1[from]) & DiagonalMask_a8h1[from]);
  Moves |= Movesa8h1[from][mask];
  mask = (B->All & RankMask[(Rank(from))]) >> (Rank(from)<<3);
  Moves |= MovesRank[from][mask];
  mask = (B->R90 & RankMask[(File(from))]) >> (File(from)<<3);
  Moves |= MovesFile[from][mask];

  return Moves;
}
