/**********************************
 *    board.c                     *
 *    Colin Frayn                 *
 *    June 2002                   *
 **********************************/

/*
  This file contains all the algorithms for messing
  about with bitboards.
*/

#include "stdafx.h"
#include "common.h"
#include "board.h"

extern int RotateR90[64],RotateL90[64];
extern int RotateR45[64],RotateL45[64],InvRotateR45[64],InvRotateL45[64];
extern BITBOARD Mask[64],FPMask1,FPMask2;
extern int first_piece[65536],last_piece[65536];

 /* Return a bitboard rotated through 90 degrees anti-clockwise */
BITBOARD RotateBoard_R90(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(RotateR90[i])]) R += (b<<i); 
  }
  return R;
}

 /* Return a bitboard rotated through 90 degrees clockwise */
BITBOARD RotateBoard_L90(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(RotateL90[i])]) R += (b<<i); 
  }
  return R;
}

 /* Return a bitboard in a1h8 diagonal form */
BITBOARD RotateBoard_R45(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(RotateR45[i])]) R += (b<<i); 
  }
  return R;
}

 /* Return a bitboard in a8h1 diagonal form */
BITBOARD RotateBoard_L45(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(RotateL45[i])]) R += (b<<i); 
  }
  return R;
}

 /* Transform back a bitboard from a1h8 diagonal form */
BITBOARD InvRotateBoard_R45(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(InvRotateR45[i])]) R += (b<<i); 
  }
  return R;
}

 /* Transform back a bitboard from a8h1 diagonal form */
BITBOARD InvRotateBoard_L45(const BITBOARD B) {
  BITBOARD R=0,b=UNIT,i;
  for (i=0;i<64;i++) {
    if (B & Mask[(InvRotateL45[i])]) R += (b<<i); 
  }
  return R;
}

/* Get the first piece, scanning horizontally then vertically from
 * the top left corner, or a8. Do this by subdividing the bitboard
 * into 4 slices by rank.  Algorithm borrowed from Dr. Robert Hyatt
 * with much thanks.  Slight modifications by CMF.  */
int FirstPiece(BITBOARD B) {
  if (B&TwoFullRanks) return first_piece[B&TwoFullRanks];
  if (B&FPMask1) return first_piece[(B >> 16)&TwoFullRanks] + 16;
  if (B&FPMask2) return first_piece[(B >> 32)&TwoFullRanks] + 32;
  return first_piece[B >> 48] + 48;
}

/* Set the board specified to the piecelist given */
void SetBoard(Board *B,int *piecelist) {
  int sq=0,p;

  B->ep = -1;
  B->side = WHITE;
  B->WhitePieces = B->BlackPieces = 0;
  B->WhitePawns = B->BlackPawns = 0;
  B->WhiteRooks = B->BlackRooks = 0;
  B->WhiteKnights = B->BlackKnights = 0;
  B->WhiteBishops = B->BlackBishops = 0;
  B->WhiteQueens = B->BlackQueens = 0;

   /* Read in the board layout */
  for (sq=0;sq<64;sq++) {
    p = B->pieces[sq] = piecelist[sq];
    if (p==0) continue;
    switch(p) {
     case (wpawn)   : B->WhitePawns |= Mask[sq]; break;
     case (wrook)   : B->WhiteRooks |= Mask[sq]; break;
     case (wknight) : B->WhiteKnights |= Mask[sq]; break;
     case (wbishop) : B->WhiteBishops |= Mask[sq]; break;
     case (wqueen)  : B->WhiteQueens |= Mask[sq]; break;
     case (wking)   : B->WhiteKing = sq; break; 
     case (bpawn)   : B->BlackPawns |= Mask[sq]; break;
     case (brook)   : B->BlackRooks |= Mask[sq]; break;
     case (bknight) : B->BlackKnights |= Mask[sq]; break;
     case (bbishop) : B->BlackBishops |= Mask[sq]; break;
     case (bqueen)  : B->BlackQueens |= Mask[sq]; break;
     case (bking)   : B->BlackKing = sq; break; 
     default        : break;
    }
    if (p>0) B->WhitePieces |= Mask[sq];
    if (p<0) B->BlackPieces |= Mask[sq];
  }

  B->All = B->WhitePieces | B->BlackPieces;
  B->R90 = RotateBoard_R90(B->All);
  B->R45 = RotateBoard_R45(B->All);
  B->L45 = RotateBoard_L45(B->All);
}

